<?php

namespace App\Livewire\Cliente;

use App\Enums\TipoClientes;
use App\Models\Cliente;
use App\Models\Empresa;
use App\Models\Persona;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Locked;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

final class ClienteComponent extends Component
{
    use LivewireAlert;
    use WithPagination;

    #[Url]
    public string $search = '';

    #[Locked]
    public string $modalTitle = '';

    #[Locked]
    public string $rifLabel = '';

    #[Locked]
    public string $razonLabel = '';

    #[Locked]
    public string $rif = '';

    #[Locked]
    public string $razon = '';

    #[Locked]
    public string $direccion = '';

    #[Locked]
    public string $telefono = '';

    #[Locked]
    public string $telefono_secundario = '';

    #[Locked]
    public string $whatsapp = '';

    #[Locked]
    public string $telegram = '';

    #[Locked]
    public string $correo = '';

    #[Locked]
    public string $tipoCliente = '';

    #[Locked]
    public string $cliente = '';

    public function render()
    {
        $search = $this->search;

        $clientes = Cliente::whereHasMorph(
            'clientable',
            [Empresa::class, Persona::class],
            function ($query, $type) use ($search) {
                if ($type === Empresa::class) {
                    $query->where('razon', 'like', "%$search%")
                        ->orWhere('rif', 'like', "%$search%")
                        ->orWhere('telefono', 'like', "%$search%")
                        ->orWhere('correo', 'like', "%$search%");
                } elseif ($type === Persona::class) {
                    $query->where('nombres', 'like', "%$search%")
                        ->orWhere('cedula', 'like', "%$search%")
                        ->orWhere('telefono', 'like', "%$search%")
                        ->orWhere('correo', 'like', "%$search%");
                }
            }
        )->paginate(15);

        if (session()->has('message')) $this->alert(session('type'), session('message'));

        return view('livewire.cliente.cliente-component', ['clientes' => $clientes]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function detail(string $uuid)
    {
        try {
            $cliente = Cliente::where('uuid', $uuid)->firstOrFail();
            $clientable = $cliente->clientable;

            switch (TipoClientes::from($cliente->clientable_type)) {
                case TipoClientes::PERSONA:
                    $this->fill([
                        'modalTitle' => "Información $clientable->nombres",
                        'razon' => $clientable->nombres,
                        'rif' => $clientable->cedula,
                        'razonLabel' => 'Nombre',
                        'rifLabel' => 'Cédula',
                    ]);
                    break;
                case TipoClientes::EMPRESA:
                    $this->fill([
                        'modalTitle' => "Información $clientable->razon",
                        'razon' => $clientable->razon,
                        'rif' => $clientable->rif,
                        'razonLabel' => 'Razón',
                        'rifLabel' => 'RIF',
                    ]);
                    break;
            }

            $this->fill([
                'direccion' => $clientable->direccion,
                'telefono' => $clientable->telefono,
                'telefono_secundario' => $clientable->telefono_secundario,
                'whatsapp' => $clientable->whatsapp,
                'telegram' => $clientable->telegram,
                'correo' => $clientable->correo,
                'tipoCliente' => TipoClientes::from($cliente->clientable_type)->getName(),
            ]);

            $this->dispatch('show-detail');
        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Cliente no encontrada.', [
                'timer' => 5000,
            ]);
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al mostrar el detalle del cliente.', [
                'timer' => 5000,
            ]);
        }
    }

    public function confirmDelete(string $uuid)
    {
        $this->cliente = $uuid;

        $this->alert('warning', '¿Estas seguro?', [
            'position' => 'center',
            'text' => '¡Si elimina a este cliente no podra restaurarlo!',
            'showConfirmButton' => true,
            'confirmButtonText' => '¡Eliminar!',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'toast' => false,
            'onConfirmed' => 'confirmedDelete',
            'onDeny' => 'deniedDelete',
            'timer' => null,
            'reverseButtons' => true,
            'confirmButtonColor' => '#3085d6',
            'cancelButtonColor' => '#d33',
        ]);
    }

    public function getListeners()
    {
        return [
            'confirmedDelete',
            'deniedDelete',
        ];
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        try {
            $cliente = Cliente::where('uuid', $this->cliente)->first();

            if ($cliente) {
                if ($cliente->remotas->count() > 0) {
                    $this->alert('error', 'No se puede eliminar al cliente porque tiene remotas asociadas');
                } else {
                    $cliente->clientable->delete();
                    $cliente->delete();
                    $this->alert(message: 'Cliente eliminado con exito');
                    $this->search = '';
                }
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al eliminar al cliente');
        }
    }

    #[On('deniedDelete')]
    public function deniedDelete()
    {
        $this->cliente = '';
    }
}
