<?php

namespace App\Livewire\Empresa;

use App\Livewire\Data\EmpresaData;
use App\Models\Empresa;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Locked;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

final class EmpresaComponent extends Component
{
    use LivewireAlert;
    use WithPagination;

    #[Url]
    public string $search = '';

    #[Locked]
    public string $empresa = '';

    #[Locked]
    public string $modalTitle = '';

    #[Locked]
    public EmpresaData $detailData;

    public function render()
    {
        $empresas = Empresa::search($this->search)->paginate(15);

        return view('livewire.empresa.empresa-component', ['empresas' => $empresas]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function detail(string $uuid)
    {
        try {
            $empresa = Empresa::where('uuid', $uuid)->firstOrFail();

            $this->fill([
                'modalTitle' => "Información $empresa->razon",
                'detailData' => EmpresaData::create($empresa->toArray()),
            ]);

            $this->dispatch('show-detail');

        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Empresa no encontrada.', [
                'timer' => 5000,
            ]);
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al mostrar el detalle de la empresa.', [
                'timer' => 5000,
            ]);
        }
    }

    public function confirmDelete(string $uuid)
    {
        $this->empresa = $uuid;

        $this->alert('warning', '¿Estas seguro?', [
            'position' => 'center',
            'text' => '¡Si elimina a esta empresa no podra restaurarla!',
            'showConfirmButton' => true,
            'confirmButtonText' => '¡Eliminar!',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'toast' => false,
            'onConfirmed' => 'confirmedDelete',
            'onDeny' => 'deniedDelete',
            'timer' => null,
            'reverseButtons' => true,
            'confirmButtonColor' => '#3085d6',
            'cancelButtonColor' => '#d33',
        ]);
    }

    public function getListeners()
    {
        return [
            'confirmedDelete',
            'deniedDelete',
        ];
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        try {
            $empresa = Empresa::where('uuid', $this->empresa)->first();

            if ($empresa) {
                $empresa->delete();
                $this->alert(message: 'Empresa eliminada con exito');
            }

        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al eliminar a la empresa');
        }
    }

    #[On('deniedDelete')]
    public function deniedDelete()
    {
        $this->empresa = '';
    }
}
