<?php

namespace App\Livewire\Forms;

use App\Models\Mikrotik;
use App\Models\ModeloMikrotik;
use App\Models\ProtocoloMikrotik;
use Illuminate\Validation\Rule;
use Livewire\Form;

final class MikrotikForm extends Form
{
    public string $nombre = '';

    public string $ip = '';

    public string $serial = '';

    public string $identify = '';

    public string $vpnUser = '';

    public string $vpnPassword = '';

    public string $vpnServer = '';

    public string $ethrCorte1 = '';

    public string $ethrCorte2 = '';

    public string $usuario = '';

    public string $clave = '';

    public string $modelo = '';

    public array $protocolos = [];

    public ?Mikrotik $mikrotik = null;

    public function rules()
    {
        $mikrotik = $this->mikrotik;

        return [
            'nombre' => ['required', Rule::unique('mikrotiks')->ignore($mikrotik)],
            'ip' => ['required', 'ip'],
            'serial' => ['required', Rule::unique('mikrotiks')->ignore($mikrotik)],
            'identify' => ['required', Rule::unique('mikrotiks')->ignore($mikrotik)],
            'vpnUser' => ['required'],
            'vpnPassword' => ['required'],
            'vpnServer' => ['required'],
            'usuario' => ['required'],
            'clave' => ['required'],
            'modelo' => ['required'],
            'protocolos.*' => ['required', 'numeric'],
        ];
    }

    public function messages()
    {

        return [
            'nombre.required' => "El campo es obligatorio.",
            'nombre.unique' => "Ya existe un mikrotik registrado con ese nombre.",
            'ip.required' => "El campo es obligatorio.",
            'ip.ip' => "Formato de IP invalido.",
            'serial.required' => "El campo es obligatorio.",
            'serial.unique' => "Ya existe un mikrotik registrado con ese serial.",
            'identify.required' => "El campo es obligatorio.",
            'identify.unique' => "Ya existe un mikrotik registrado con ese identify.",
            'vpnUser.required' => "El campo es obligatorio.",
            'vpnPassword.required' => "El campo es obligatorio.",
            'vpnServer.required' => "El campo es obligatorio.",
            'usuario.required' => "El campo es obligatorio.",
            'clave.required' => "El campo es obligatorio.",
            'modelo.required' => "El campo es obligatorio.",
            'protocolos.*.required' => "El campo es obligatorio.",
            'protocolos.*.numeric' => "El protocolo debe contener solo números."
        ];
    }

    public function save(?Mikrotik $mikrotik = null)
    {

        $this->mikrotik = $mikrotik;
        $this->validate();

        if ($mikrotik) {
            $this->update($mikrotik);
        } else {

            return $this->store();
        }
    }

    private function store(): Mikrotik
    {

        $mikrotik = Mikrotik::create([
            'nombre' => $this->nombre,
            'ip' => $this->ip,
            'serial' => $this->serial,
            'identify' => $this->identify,
            'vpn_user' => $this->vpnUser,
            'vpn_password' => $this->vpnPassword,
            'vpn_server' => $this->vpnServer,
            'usuario' => $this->usuario,
            'clave' => $this->clave,
            'modelo_mikrotik_id' => ModeloMikrotik::where('uuid', $this->modelo)->value('id'),
        ]);

        $nombres_protocolos = array_keys($this->protocolos);
        $protocolos = ProtocoloMikrotik::whereIn('protocolo', $nombres_protocolos)->get();

        $protocolosSync = [];
        foreach ($this->protocolos as $index => $protocoloInput) {
            $protocolo = $protocolos->firstWhere('protocolo', $index);
            if ($protocolo) {
                $protocolosSync[$protocolo->id] = ['puerto' => $protocoloInput];
            }
        }

        $mikrotik->protocolos()->sync($protocolosSync);

        return $mikrotik;
    }

    private function update(Mikrotik $mikrotik)
    {

        $mikrotik->fill([
            'nombre' => $this->nombre,
            'ip' => $this->ip,
            'serial' => $this->serial,
            'identify' => $this->identify,
            'vpn_user' => $this->vpnUser,
            'vpn_password' => $this->vpnPassword,
            'vpn_server' => $this->vpnServer,
            'usuario' => $this->usuario,
            'clave' => $this->clave,
            'modelo_mikrotik_id' => ModeloMikrotik::where('uuid', $this->modelo)->value('id'),
        ]);

        $nombres_protocolos = array_keys($this->protocolos);
        $protocolos = ProtocoloMikrotik::whereIn('protocolo', $nombres_protocolos)->get();

        foreach ($this->protocolos as $index => $protocoloInput) {
            $protocolo = $protocolos->firstWhere('protocolo', $index);
            if ($protocolo) {
                $mikrotik->protocolos()->updateExistingPivot($protocolo->id, ['puerto' => $protocoloInput]);
            }
        }

        $mikrotik->save();
    }
}
