<?php

namespace App\Livewire\Mikrotik;

use App\Livewire\Data\MikrotikData;
use App\Models\Mikrotik;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Locked;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

final class MikrotikComponent extends Component
{
    use LivewireAlert;
    use WithPagination;

    #[Url]
    public string $search = '';

    #[Locked]
    public string $modalTitle = '';

    #[Locked]
    public MikrotikData $detailData;

    #[Locked]
    public string $mikrotik = '';

    public function render()
    {
        $mikrotiks = Mikrotik::search($this->search)->paginate(15);
        if (session()->has('message')) $this->alert(session('type'), session('message'));

        return view('livewire.mikrotik.mikrotik-component', ['mikrotiks' => $mikrotiks]);
    }

    public function mount()
    {
        $this->detailData = new MikrotikData();
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function detail(string $uuid)
    {

        try {

            $mikrotik = Mikrotik::with(['modelo', 'protocolos'])->where('uuid', $uuid)->firstOrFail();

            $this->fill([
                'detailData' => MikrotikData::create($mikrotik->toArray()),
                'modalTitle' => "Detalle Mikrotik $mikrotik->nombre",
            ]);

            $this->dispatch('show-detail');
        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Mikrotik no encontrado.');
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG', $e));
            $this->alert('error', 'Error al mostrar el detalle del mikrotik.');
        }
    }

    public function confirmDelete(string $uuid)
    {
        $this->mikrotik = $uuid;

        $this->alert('warning', '¿Estas seguro?', [
            'position' => 'center',
            'text' => '¡Si elimina este mikrotik no podra restaurarlo!',
            'showConfirmButton' => true,
            'confirmButtonText' => '¡Eliminar!',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'toast' => false,
            'onConfirmed' => 'confirmedDelete',
            'onDeny' => 'deniedDelete',
            'timer' => null,
            'reverseButtons' => true,
            'confirmButtonColor' => '#3085d6',
            'cancelButtonColor' => '#d33',
        ]);
    }

    public function getListeners()
    {
        return [
            'confirmedDelete',
            'deniedDelete',
        ];
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        try {
            $mikrotik = Mikrotik::where('uuid', $this->mikrotik)->first();

            if ($mikrotik) {
                $mikrotik->delete();
                $this->alert(message: 'Mikrotik eliminado con exito.');
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al eliminar el mikrotik.');
        }
    }

    #[On('deniedDelete')]
    public function deniedDelete()
    {
        $this->mikrotik = '';
    }
}
