<?php

namespace App\Livewire\Persona;

use App\Livewire\Data\PersonaData;
use App\Models\Persona;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Locked;
use Livewire\Attributes\On;
use Livewire\Attributes\Url;
use Livewire\Component;
use Livewire\WithPagination;

final class PersonaComponent extends Component
{
    use LivewireAlert;
    use WithPagination;

    #[Url]
    public string $search = '';

    #[Locked]
    public string $persona = '';

    #[Locked]
    public string $modalTitle = '';

    #[Locked]
    public  $detailData;
    // public PersonaData $detailData;

    public function render()
    {
        $personas = Persona::search($this->search)->paginate(15);

        return view('livewire.persona.persona-component', ['personas' => $personas]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function detail(string $uuid)
    {
        try {
            $persona = Persona::where('uuid', $uuid)->firstOrFail();

            $this->fill([
                'modalTitle' => "Información $persona->nombres",
                'detailData' => $persona,
                // 'detailData' => PersonaData::create($persona->toArray()),
            ]);

            $this->dispatch('show-detail');
        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Persona no encontrada.', [
                'timer' => 5000,
            ]);
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al mostrar el detalle de la persona.', [
                'timer' => 5000,
            ]);
        }
    }

    public function confirmDelete(string $uuid)
    {
        $this->persona = $uuid;

        $this->alert('warning', '¿Estas seguro?', [
            'position' => 'center',
            'text' => '¡Si elimina a esta persona no podra restaurarla!',
            'showConfirmButton' => true,
            'confirmButtonText' => '¡Eliminar!',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'toast' => false,
            'onConfirmed' => 'confirmedDelete',
            'onDeny' => 'deniedDelete',
            'timer' => null,
            'reverseButtons' => true,
            'confirmButtonColor' => '#3085d6',
            'cancelButtonColor' => '#d33',
        ]);
    }

    public function getListeners()
    {
        return [
            'confirmedDelete',
            'deniedDelete',
        ];
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        try {
            $persona = Persona::where('uuid', $this->persona)->first();

            if ($persona) {
                $persona->delete();
                $this->alert(message: 'Persona eliminada con exito');
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al eliminar a la persona');
        }
    }

    #[On('deniedDelete')]
    public function deniedDelete()
    {
        $this->persona = '';
    }
}
