<?php

namespace App\Livewire\Proveedor;

use App\Livewire\Data\EmpresaData;
use App\Livewire\Data\ProveedorData;
use App\Models\Empresa;
use App\Models\Proveedor;
use Exception;
use Illuminate\Database\Eloquent\ModelNotFoundException;
use Jantinnerezo\LivewireAlert\LivewireAlert;
use Livewire\Attributes\Locked;
use Livewire\Attributes\On;
use Livewire\Component;
use Livewire\WithPagination;

final class ProveedorComponent extends Component
{
    use LivewireAlert;
    use WithPagination;

    public string $search = '';

    public string $empresa = '';

    #[Locked]
    public string $modalTitle = '';

    #[Locked]
    public string $razon = '';

    #[Locked]
    public string $rif = '';

    #[Locked]
    public string $direccion = '';

    #[Locked]
    public string $telefono = '';

    #[Locked]
    public string $telefono_secundario;

    #[Locked]
    public string $whatsapp = '';

    #[Locked]
    public string $telegram = '';

    #[Locked]
    public string $correo = '';

    #[Locked]
    public ProveedorData $detailData;
    // public $detailData;


    public function render()
    {
        $search = $this->search;

        $proveedores = Proveedor::whereHas('empresa', function ($query) use ($search) {
            $query->where('razon', 'like', "%$search%")
                ->orWhere('rif', 'like', "%$search%")
                ->orWhere('telefono', 'like', "%$search%")
                ->orWhere('correo', 'like', "%$search%");
        })->paginate(15);

        if (session()->has('message')) $this->alert(session('type'), session('message'));

        return view('livewire.proveedor.proveedor-component', ['proveedores' => $proveedores]);
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function detail(string $uuid)
    {
        // dd(Empresa::where('uuid', $uuid)->firstOrFail());
        try {
            $empresa = Empresa::where('uuid', $uuid)->firstOrFail();

            $this->fill([
                'modalTitle' => "Información $empresa->razon",
                'detailData' => new ProveedorData(EmpresaData::create($empresa->toArray())),
                // 'detailData' => $empresa

            ]);

            $this->dispatch('show-detail');
        } catch (ModelNotFoundException $e) {
            $this->alert('error', 'Proveedor no encontrada.', [
                'timer' => 5000,
            ]);
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al mostrar el detalle de la proveedor.', [
                'timer' => 5000,
            ]);
        }
    }

    public function confirmDelete(string $uuid)
    {
        $this->empresa = $uuid;

        $this->alert('warning', '¿Estas seguro?', [
            'position' => 'center',
            'text' => '¡Si elimina a esta empresa no podra restaurarla!',
            'showConfirmButton' => true,
            'confirmButtonText' => '¡Eliminar!',
            'showCancelButton' => true,
            'cancelButtonText' => 'Cancelar',
            'toast' => false,
            'onConfirmed' => 'confirmedDelete',
            'onDeny' => 'deniedDelete',
            'timer' => null,
            'reverseButtons' => true,
            'confirmButtonColor' => '#3085d6',
            'cancelButtonColor' => '#d33',
        ]);
    }

    public function getListeners()
    {
        return [
            'confirmedDelete',
            'deniedDelete',
        ];
    }

    #[On('confirmedDelete')]
    public function confirmedDelete()
    {
        try {
            $empresa = Empresa::with('proveedor')->where('uuid', $this->empresa)->first();

            if ($empresa->proveedor) {
                if ($empresa->proveedor->remotas->count() > 0) {
                    $this->alert('error', 'No se puede eliminar al proveedor, tiene remotas asociadas.');
                } else {
                    $empresa->proveedor->delete();
                    $this->alert(message: 'Proveedor eliminado con exito.');
                    $this->search = '';
                }
            }
        } catch (Exception $e) {
            throw_if(env('APP_DEBUG'), $e);
            $this->alert('error', 'Error al eliminar al proveedor.');
        }
    }

    #[On('deniedDelete')]
    public function deniedDelete()
    {
        $this->empresa = '';
    }
}
