<?php

namespace App\Models;

use App\Traits\GenerateUuid;
use App\Traits\RouteUuid;
use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Facades\DB;
use App\Models\CobrosHistorial;
use App\Models\PaymentMethod;

class Cobranza extends Model
{
    // use GenerateUuid;
    use HasFactory;
    // use RouteUuid;

    protected $table = 'cobros';

    protected $fillable = [
        'cliente_id',
        'concepto',
        'remota_id',
        'monto',
        'fecha',
    ];

    public function cliente(): BelongsTo
    {
        return $this->belongsTo(Cliente::class);
    }

    public function remota(): BelongsTo
    {
        return $this->belongsTo(Remota::class);
    }
    public function cobroshistorial()
    {
        return $this->hasMany(CobrosHistorial::class, 'cobro_id');
    }

    // public function getFechaCorteAttribute()
    // {
    //     $fecha = $this->fecha;
    //     if ($fecha) {
    //         $fecha = new \DateTime($fecha);
    //         $fecha = $fecha->format('d/m/Y');
    //         return $fecha;
    //     } else {
    //         return null;
    //     }
    // }


    // ? Join Remotas une los campos necesarios entre cobros remotas y clientes(empresas y personas),
    // ojo pelao con mover esto
    public static function JoinRemotas($search = '', $status_cobros = null, $status_remota = null, $orderBy = 'asc')
    {
        DB::statement("SET lc_time_names = 'es_ES'");

        $query = self::with([
            'remota' => function ($query) {
                $query->with([
                    'cliente.persona',
                    'cliente.empresa',
                    'plan',
                    'socio.persona',
                    'vendedor.persona'
                ]);
            },
            'cliente.persona',
            'cliente.empresa'
        ])
            ->join('remotas', 'cobros.remota_id', '=', 'remotas.id')
            ->leftJoin('clientes', 'cobros.cliente_id', '=', 'clientes.id')
            ->leftJoin('personas', function ($join) {
                $join->on('clientes.clientable_id', '=', 'personas.id')
                    ->where('clientes.clientable_type', '=', 'App\Models\Persona');
            })
            ->leftJoin('empresas', function ($join) {
                $join->on('clientes.clientable_id', '=', 'empresas.id')
                    ->where('clientes.clientable_type', '=', 'App\Models\Empresa');
            })->select(
                'cobros.*',
                'cobros.id              as cobro_id',
                'cobros.cliente_id      as cobro_cliente_id',
                'cobros.concepto        as concepto',
                'cobros.status          as cobro_status',
                'cobros.monto           as cobro_monto',
                DB::raw('DATE_FORMAT(cobros.fecha, "%d/ %m/ %Y") as cobro_fecha'),
                'remotas.id             as remota_id',
                'remotas.dia_corte      as remota_dia_corte',
                'remotas.equipo         as remota_equipo',
                'remotas.meses_contrato as remota_meses_contrato',
                'remotas.status         as remota_status',
                DB::raw('COALESCE(personas.nombres, empresas.razon) as cliente_nombre'),
                DB::raw('DATE_FORMAT(DATE_ADD(remotas.dia_corte, INTERVAL 1 MONTH), "%d/ %m/ %Y") as proxima_fecha_cobro'),
                DB::raw('DATE_FORMAT(DATE_ADD(remotas.dia_corte, INTERVAL 1 MONTH), "%Y-%m-%d") as proxima_fecha_cobro_date'),
                DB::raw('MONTH(cobros.fecha) as mes_cobro'),
                DB::raw('DATE_FORMAT(cobros.fecha, "%M") as mes_cobro_nombre'),
                DB::raw('ABS(DATEDIFF(NOW(), cobros.fecha)) as dias_diferencia'), // Usar ABS para que sea siempre positivo
                DB::raw('COALESCE((SELECT SUM(monto) FROM cobros_historial WHERE cobro_id = cobros.id), 0) as cantidad_pagada'),
                DB::raw('cobros.monto - COALESCE((SELECT SUM(monto) FROM cobros_historial WHERE cobro_id = cobros.id), 0) as cantidad_pendiente'),
                DB::raw('CASE WHEN remotas.TxTIdentity IS NOT NULL THEN 1 ELSE 0 END as tiene_licencia_asociada')
            );

        if ($status_cobros) {
            $query->where('cobros.status', '=', $status_cobros);
        }
        if ($status_remota) {
            $query->where('remotas.status', '=', $status_remota);
        }
        if ($search) {
            $query->where(function ($query) use ($search) {
                $query->where('personas.nombres', 'LIKE', '%' . $search . '%')
                    ->orWhere('empresas.razon', 'LIKE', '%' . $search . '%')
                    ->orWhere('remotas.equipo', 'LIKE', '%' . $search . '%')
                    ->orWhere('remotas.correo_cuenta', 'LIKE', '%' . $search . '%')
                    ->orWhere('remotas.kit_serial', 'LIKE', '%' . $search . '%')
                    ->orWhere('remotas.antena_serial', 'LIKE', '%' . $search . '%');
            });
        }

        $query->orderBy('fecha', $orderBy);

        return $query;
    }

    public static function montoCobrosParticularesPendiente()
    {
        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        $cobranzas = self::where('status', 'pendiente')
            ->where('concepto', '!=', 'Pago de mensualidad')
            // ->where(function ($query) use ($currentMonth, $currentYear) {
            //     $query->whereMonth('created_at', '<=', $currentMonth)
            //         ->whereYear('created_at', $currentYear);
            // })
            ->where('fecha', '<', Carbon::now())

            ->get();

        $totalMonto = $cobranzas->sum('monto');

        return round($totalMonto, 2); // Devolver el total del monto
    }
    public static function montoCobrosMensualidadesPendiente()
    {
        // $currentMonth = Carbon::now()->month;
        // $currentYear = Carbon::now()->year;

        $cobranzas = self::where('status', 'pendiente')
            ->where('concepto', 'Pago de mensualidad')
            ->whereMonth('fecha', Carbon::now()->month)
            ->whereYear('fecha', Carbon::now()->year)
            ->get();

        $totalMonto = $cobranzas->sum('monto');

        return round($totalMonto, 2); // Devolver el total del monto
    }
    public static function montoCobrosMensualidadesProximosNdias($dias)
    {
        $currentMonth = Carbon::now()->month;
        $currentYear = Carbon::now()->year;

        $cobranzas = self::where('status', 'pendiente')
            ->where('concepto', 'Pago de mensualidad')
            ->where('fecha', '>=', Carbon::now()) // Desde hoy
            ->where('fecha', '<=', Carbon::now()->addDays($dias)) // Hasta los próximos $dias
            ->get();

        $totalMonto = $cobranzas->sum('monto');

        return round($totalMonto, 2); // Devolver el total del monto
    }


    public function tienePagosRegistrados()
    {
        return $this->cobroshistorial->count() > 0;
    }
    public function getFechaDeCobroAttribute()
    {
        $fecha = $this->fecha;
        if ($fecha) {
            $fecha = new \DateTime($fecha);
            $fecha = $fecha->format('d/m/Y');
            return $fecha;
        } else {
            return null;
        }
    }
    public function getMesDeCobroAttribute()
    {
        $fecha = $this->fecha; // Asumiendo que $this->fecha es una cadena de fecha válida
        if ($fecha) {
            // Crear una instancia de Carbon a partir de la fecha
            $fecha = Carbon::createFromFormat('Y-m-d', $fecha); // Ajusta el formato según sea necesario

            // Obtener el nombre del mes en español
            return $fecha->translatedFormat('F'); // 'F' devuelve el nombre completo del mes
        } else {
            return null;
        }
    }
    public function getMontosPorMetodoDePago($fecha_desde = null, $fecha_hasta = null)
    {
        // Obtener todos los cobros del historial con sus métodos de pago
        $query = CobrosHistorial::with('paymentMethod')
            ->select('payment_method_id', DB::raw('COUNT(*) as cantidad'), DB::raw('ROUND(SUM(monto), 2) as monto_total'))
            ->whereNotNull('payment_method_id');

        // Aplicar filtro de fechas si se proporcionan
        if ($fecha_desde && $fecha_hasta) {
            $query->whereBetween('fecha_pago', [$fecha_desde, $fecha_hasta]);
        }

        $cobrosHistorial = $query->groupBy('payment_method_id')->get();

        // Transformar los datos al formato requerido
        $metodosPago = [];
        foreach ($cobrosHistorial as $cobro) {
            $metodosPago[] = [
                'nombre' => $cobro->paymentMethod ? $cobro->paymentMethod->getName() : 'Sin método',
                'cantidad' => $cobro->cantidad,
                'monto' => round($cobro->monto_total, 2)
            ];
        }

        return $metodosPago;
    }
    // \Carbon\Carbon::parse($remota->fecha_de_cobro)->locale('es')->format('F')??''

    public function diasPasados()
    {
        $fechaCobranza = Carbon::parse($this->fecha);
        return Carbon::now()->diffInDays($fechaCobranza);
    }
    // buscar en el historial de cobros , cobros con el id
    public  function cantidadPagadaDelUltimoCobroPendiente()
    {
        // $cobros = CobrosHistorial::where('cobro_id', $this->id)->get();
        $cobro = Cobranza::find($this->id);

        $hasCobrosRealizados = count($cobro->cobroshistorial) > 0 ? true : false;
        if ($hasCobrosRealizados) {
            $cobrosHistorial = $cobro->cobroshistorial;
            // sumar los montos de los cobros
            $total = 0;
            foreach ($cobrosHistorial as $cobro) {
                $total += (float)$cobro->monto;
            }
            return $total;
            # code...
        } else {
            return 0;
        }
    }
    public function cantidadPendienteDelUltimoCobroPendiente()
    {
        return $this->monto - $this->cantidadPagadaDelUltimoCobroPendiente();
    }

    public function tieneSaldoPendiente()
    {
        return $this->monto > $this->cantidadPagadaDelUltimoCobroPendiente();
    }
    public function utilidadActual()
    {
        return $this->cantidadPagadaDelUltimoCobroPendiente() - $this->remota->getCostoDeVenta();
    }

    public static function cuentasPorCobrar()
    {
        return self::where('status', 'pendiente')
            ->where('fecha', '<', Carbon::now()->endOfMonth())
            ->where('fecha', '>', Carbon::now()->startOfMonth())
            ->count();
    }

    public static function countCuentasCobradas()
    {
        return self::whereHas('remota', function ($query) {
            $query->where('status', 'activo');
        })
            ->where('status', 'pagado')
            ->where('fecha', '>', Carbon::now()->startOfMonth())
            ->where('fecha', '<', Carbon::now()->endOfMonth())
            ->count();
    }
    public static function saldoPorCobrar()
    {
        $total = self::where('status', 'pendiente')
            ->where('fecha', '>', Carbon::now()->startOfMonth())
            ->where('fecha', '<', Carbon::now())
            // ->where('fecha', '<', Carbon::now()->endOfMonth())
            ->sum('monto');

        return round($total, 2);
    }
    public static function getPagosMensualidad()
    {
        return self::where('concepto', 'Pago de mensualidad')->where('status', 'pagado')->get();
    }
    // Suma de montos de pagos de mensualidad
    public static function totalPagosMensualidad($fecha_desde, $fecha_hasta, $status_cobro = 'pagado')
    {
        $total = self::where('concepto', 'Pago de mensualidad')
            ->where('fecha', '>=', $fecha_desde)
            ->where('fecha', '<=', $fecha_hasta)
            ->where('status', $status_cobro)
            ->sum('monto');

        return round($total, 2);
    }
    // pagos que no son mensualidad
    public static function getPagosNoMensualidad()
    {
        return self::where('concepto', '!=', 'Pago de mensualidad')->get();
    }
    // Suma de montos de pagos que no son mensualidad
    public static function totalPagosNoMensualidad($fecha_desde, $fecha_hasta,  $status_cobro = 'pagado')
    {
        $total = self::where('concepto', '!=', 'Pago de mensualidad')
            ->where('fecha', '>=', $fecha_desde)
            ->where('fecha', '<=', $fecha_hasta)
            ->where('status', $status_cobro)
            ->sum('monto');

        return round($total, 2);
    }
    public static function saldoCobradoRemotasActivasMesActual()
    {
        // return self::where('status', 'pagado')
        //     ->where('updated_at', '>', Carbon::now()->startOfMonth())
        //     ->where('updated_at', '<', Carbon::now()->endOfMonth())
        //     ->whereHas('remota', function ($query) {
        //         $query->where('status', 'activo');
        //     })
        //     ->sum('monto');
        // Obtener el inicio y el fin del mes actual
        $inicioMes = Carbon::now()->startOfMonth();
        $finMes = Carbon::now()->endOfMonth();

        // Realizar la consulta
        $totalCobros = CobrosHistorial::where('fecha_pago', '>=', $inicioMes)
            ->where('fecha_pago', '<=', $finMes)
            ->whereHas('remota', function ($query) {
                $query->where('status', 'activo'); // Verificar que remota esté activa
            })
            ->sum('monto');

        return round($totalCobros, 2);
    }
    public static function saldoCobradoRemotas($fecha_desde = null, $fecha_hasta = null, $status = 'activo')
    {
        $totalCobros = CobrosHistorial::where('fecha_pago', '>=', $fecha_desde)
            ->where('fecha_pago', '<=', $fecha_hasta)
            ->whereHas('remota', function ($query) use ($status) {
                $query->where('status', $status); // Verificar que remota esté activa
            })
            ->sum('monto');

        return round($totalCobros, 2);
    }

    // totalPagadoMensualidades
    public static function totalPagadoMensualidades($fecha_desde, $fecha_hasta, $status_cobro = 'pagado')
    {
        $total = self::where('concepto', 'Pago de mensualidad')
            ->where('fecha', '>=', $fecha_desde)
            ->where('fecha', '<=', $fecha_hasta)
            ->where('status', $status_cobro)
            ->sum('monto');

        return round($total, 2);
    }
    // totalCostoMensualidades
    public static function totalCostoMensualidades($fecha_desde, $fecha_hasta)
    {


        $total = self::where('concepto', 'Pago de mensualidad')
            ->where('status', 'pagado') // Solo cobranzas pagadas
            ->with('remota')
            ->with('remota.plan')
            ->where('fecha', '>=', $fecha_desde)
            ->where('fecha', '<=', $fecha_hasta)
            ->whereHas('remota', function ($query) {
                $query->select('id', 'costo'); // Selecciona las columnas necesarias de Remota
            })
            ->get()
            ->sum(function ($cobranza) {
                return round($cobranza->remota->plan->costo, 2); // Suma los costos de las remotas asociadas
            });

        return round($total, 2);
    }
    // totalComisionesMensualidades
    public static function totalComisionesMensualidades($fecha_desde, $fecha_hasta)
    {
        $total = self::where('concepto', 'Pago de mensualidad')
            ->where('status', 'pagado') // Solo cobranzas pagadas
            ->with('remota')
            ->where('fecha', '>=', $fecha_desde)
            ->where('fecha', '<=', $fecha_hasta)
            ->whereHas('remota', function ($query) {
                $query->select('id', 'costo', 'comision'); // Selecciona las columnas necesarias de Remota
            })
            ->get()
            ->sum(function ($cobranza) {
                return round($cobranza->remota->comision, 2); // Suma las comisiones de las remotas asociadas
            });

        return round($total, 2);
    }
    // totalUtilidadMensualidades
    public static function totalUtilidadMensualidades($fecha_desde, $fecha_hasta)
    {
        $total = self::whereHas('cobroshistorial', function ($query) use ($fecha_desde, $fecha_hasta) {
            $query->where('fecha', '>=', $fecha_desde)
                ->where('fecha', '<=', $fecha_hasta)
                ->where('status', 'pagado'); // Asegúrate de que 'status' sea la columna correcta
        })
            ->with('cobroshistorial') // Carga los historiales para sumar la utilidad
            ->get()
            ->sum(function ($cobranza) {
                return $cobranza->cobroshistorial->sum(function ($historial) {
                    return round($historial->utilidad, 2);
                });
            });

        return round($total, 2);
    }
    public static function obtenerSumaMontosPagadosPorProveedor($fecha_desde, $fecha_hasta, $proveedorId)
    {
        $total = self::where('status', 'pagado') // Filtrar por estado 'pagado'
            ->whereBetween('fecha', [$fecha_desde, $fecha_hasta]) // Filtrar por rango de fechas
            ->whereHas('remota', function ($query) use ($proveedorId) {
                $query->where('proveedor_id', $proveedorId); // Filtrar por proveedor_id
            })
            ->sum('monto'); // Cambia 'monto' al nombre de la columna que contiene el monto

        return round($total, 2);
    }

    public static function totalCobranzas()
    {
        $total = self::where('status', 'pagado')->sum('monto');
        return round($total, 2);
    }

    /**
     * Obtiene los totales mensuales de cobranzas para un año específico
     * con cálculo de variación porcentual mes a mes, utilidad bruta y utilidad neta
     */
    public static function getTotalesMensuales($year = null)
    {
        if (!$year) {
            $year = Carbon::now()->year;
        }

        // Obtener totales por mes para el año especificado con costos y comisiones
        $totalesMensuales = self::where('status', 'pagado')
            ->whereYear('fecha', $year)
            ->with(['remota.plan'])
            ->selectRaw('MONTH(fecha) as mes, ROUND(SUM(monto), 2) as total')
            ->groupBy('mes')
            ->orderBy('mes')
            ->get()
            ->keyBy('mes');

        // Obtener costos y comisiones por mes
        $costosMensuales = self::where('status', 'pagado')
            ->whereYear('fecha', $year)
            ->with(['remota.plan'])
            ->get()
            ->groupBy(function ($cobranza) {
                return Carbon::parse($cobranza->fecha)->month;
            })
            ->map(function ($cobranzas) {
                $totalCostos = 0;
                $totalComisiones = 0;

                foreach ($cobranzas as $cobranza) {
                    // Usar el costo del plan si está disponible, sino usar el costo de la remota
                    $costo = $cobranza->remota->plan->costo ?? $cobranza->remota->costo ?? 0;
                    $comision = $cobranza->remota->comision ?? 0;

                    $totalCostos += round($costo, 2);
                    $totalComisiones += round($comision, 2);
                }

                return [
                    'costos' => round($totalCostos, 2),
                    'comisiones' => round($totalComisiones, 2)
                ];
            });

        // Crear array con todos los meses del año
        $meses = [
            1 => 'Enero',
            2 => 'Febrero',
            3 => 'Marzo',
            4 => 'Abril',
            5 => 'Mayo',
            6 => 'Junio',
            7 => 'Julio',
            8 => 'Agosto',
            9 => 'Septiembre',
            10 => 'Octubre',
            11 => 'Noviembre',
            12 => 'Diciembre'
        ];

        $resultado = [];
        $mesAnterior = null;
        $utilidadBrutaAnterior = null;
        $utilidadNetaAnterior = null;

        foreach ($meses as $numeroMes => $nombreMes) {
            $total = $totalesMensuales->get($numeroMes);
            $monto = $total ? round($total->total, 2) : 0;

            // Obtener costos y comisiones del mes
            $costosMes = $costosMensuales->get($numeroMes);
            $costos = $costosMes ? round($costosMes['costos'], 2) : 0;
            $comisiones = $costosMes ? round($costosMes['comisiones'], 2) : 0;

            // Calcular utilidades
            $utilidadBruta = round($monto - $costos, 2); // Total facturado - costos del plan
            $utilidadNeta = round($monto - $costos - $comisiones, 2); // Total facturado - costos - comisiones

            // Calcular variación porcentual para total
            $variacion = null;
            if ($mesAnterior !== null && $mesAnterior > 0) {
                $variacion = (($monto - $mesAnterior) / $mesAnterior) * 100;
            }

            // Calcular variación porcentual para utilidad bruta
            $variacionUtilidadBruta = null;
            if ($utilidadBrutaAnterior !== null && $utilidadBrutaAnterior != 0) {
                $variacionUtilidadBruta = (($utilidadBruta - $utilidadBrutaAnterior) / abs($utilidadBrutaAnterior)) * 100;
            }

            // Calcular variación porcentual para utilidad neta
            $variacionUtilidadNeta = null;
            if ($utilidadNetaAnterior !== null && $utilidadNetaAnterior != 0) {
                $variacionUtilidadNeta = (($utilidadNeta - $utilidadNetaAnterior) / abs($utilidadNetaAnterior)) * 100;
            }

            $resultado[] = [
                'mes' => $nombreMes,
                'numero_mes' => $numeroMes,
                'total' => $monto,
                'utilidad_bruta' => $utilidadBruta,
                'utilidad_neta' => $utilidadNeta,
                'comisiones' => $comisiones,
                'variacion' => $variacion,
                'variacion_formateada' => $variacion !== null
                    ? ($variacion >= 0 ? '+' . number_format($variacion, 1) . '%' : number_format($variacion, 1) . '%')
                    : '–',
                'variacion_utilidad_bruta' => $variacionUtilidadBruta,
                'variacion_utilidad_bruta_formateada' => $variacionUtilidadBruta !== null
                    ? ($variacionUtilidadBruta >= 0 ? '+' . number_format($variacionUtilidadBruta, 1) . '%' : number_format($variacionUtilidadBruta, 1) . '%')
                    : '–',
                'variacion_utilidad_neta' => $variacionUtilidadNeta,
                'variacion_utilidad_neta_formateada' => $variacionUtilidadNeta !== null
                    ? ($variacionUtilidadNeta >= 0 ? '+' . number_format($variacionUtilidadNeta, 1) . '%' : number_format($variacionUtilidadNeta, 1) . '%')
                    : '–'
            ];

            $mesAnterior = $monto;
            $utilidadBrutaAnterior = $utilidadBruta;
            $utilidadNetaAnterior = $utilidadNeta;
        }

        return $resultado;
    }

    /**
     * Calcula el rendimiento del mes actual vs mes anterior
     */
    public static function calcularRendimientoActual()
    {
        $mesActual = Carbon::now()->month;
        $añoActual = Carbon::now()->year;

        // Total del mes actual
        $totalMesActual = round(self::where('status', 'pagado')
            ->whereYear('fecha', $añoActual)
            ->whereMonth('fecha', $mesActual)
            ->sum('monto'), 2);

        // Total del mes anterior
        $mesAnterior = $mesActual == 1 ? 12 : $mesActual - 1;
        $añoAnterior = $mesActual == 1 ? $añoActual - 1 : $añoActual;

        $totalMesAnterior = round(self::where('status', 'pagado')
            ->whereYear('fecha', $añoAnterior)
            ->whereMonth('fecha', $mesAnterior)
            ->sum('monto'), 2);

        if ($totalMesAnterior == 0) {
            return [
                'rendimiento' => 0,
                'total_actual' => $totalMesActual,
                'total_anterior' => $totalMesAnterior
            ];
        }

        $rendimiento = round((($totalMesActual - $totalMesAnterior) / $totalMesAnterior) * 100, 2);

        return [
            'rendimiento' => $rendimiento,
            'total_actual' => $totalMesActual,
            'total_anterior' => $totalMesAnterior
        ];
    }
}
